#!/usr/bin/env zsh
set -euo nounset

#############################################
# Arch Linux install non-interactif (UEFI)
# - ESP montée sur /boot de 2G (FAT32)
# - Wi-Fi via wlan0 (SSID/PSK en variables)
#
# ATTENTION: CE SCRIPT EFFACE LE DISQUE.
# À lancer depuis l'ISO Arch (archiso) en root.
#############################################

### ====== VARIABLES À REMPLIR ======
DISK="/dev/nvme0n1"          # ex: /dev/sda ou /dev/nvme0n1
HOSTNAME="arch"
USERNAME="user"
USER_PASSWORD="changeme"
ROOT_PASSWORD="changeme"

WIFI_INTERFACE="wlan0"
WIFI_SSID="TON_SSID"
WIFI_PSK="TON_PSK"

TIMEZONE="Europe/Paris"
LOCALE="fr_FR.UTF-8"
KEYMAP="fr"
### =================================

die() {
  print -u2 "ERROR: $*"
  exit 1
}

need() {
  command -v "$1" >/dev/null 2>&1 || die "Commande manquante: $1"
}

for c in sgdisk mkfs.fat mkfs.ext4 mount umount pacstrap genfstab arch-chroot iwctl; do
  need "$c"
done

[[ -b "$DISK" ]] || die "Disque invalide: $DISK"

print "==> Vérification UEFI..."
[[ -d /sys/firmware/efi ]] || die "UEFI requis"

print "==> Synchronisation de l'heure..."
timedatectl set-ntp true 2>/dev/null || true

print "==> Connexion Wi-Fi ($WIFI_INTERFACE)..."
iwctl --passphrase "$WIFI_PSK" station "$WIFI_INTERFACE" connect "$WIFI_SSID" \
  || die "Connexion Wi-Fi échouée"

ping -c 1 archlinux.org >/dev/null 2>&1 || die "Pas d'accès réseau"

print "==> Partitionnement GPT sur $DISK"
swapoff -a 2>/dev/null || true
umount -R /mnt 2>/dev/null || true

PART_PREFIX="$DISK"
[[ "$DISK" == *nvme* ]] && PART_PREFIX="${DISK}p"

BOOT_PART="${PART_PREFIX}1"
ROOT_PART="${PART_PREFIX}2"

sgdisk --zap-all "$DISK"
sgdisk -n 1:0:+2G  -t 1:ef00 -c 1:"EFI_SYSTEM" "$DISK"
sgdisk -n 2:0:0    -t 2:8300 -c 2:"ARCH_ROOT"  "$DISK"
partprobe "$DISK"
sleep 2

print "==> Formatage..."
mkfs.fat -F32 -n BOOT "$BOOT_PART"
mkfs.ext4 -F -L ROOT "$ROOT_PART"

print "==> Montage..."
mount "$ROOT_PART" /mnt
mkdir -p /mnt/boot
mount "$BOOT_PART" /mnt/boot

print "==> Installation du système de base..."
pacstrap -K /mnt \
  base linux linux-firmware \
  networkmanager iwd sudo vim

print "==> Génération fstab..."
genfstab -U /mnt >> /mnt/etc/fstab

print "==> Configuration système (chroot)..."
arch-chroot /mnt /usr/bin/env zsh <<EOF
set -euo nounset

ln -sf /usr/share/zoneinfo/$TIMEZONE /etc/localtime
hwclock --systohc

sed -i 's/^#\\($LOCALE\\)/\\1/' /etc/locale.gen
locale-gen
print "LANG=$LOCALE" > /etc/locale.conf
print "KEYMAP=$KEYMAP" > /etc/vconsole.conf

print "$HOSTNAME" > /etc/hostname
cat >/etc/hosts <<H
127.0.0.1 localhost
::1       localhost
127.0.1.1 $HOSTNAME.localdomain $HOSTNAME
H

print "root:$ROOT_PASSWORD" | chpasswd
useradd -m -G wheel -s /bin/zsh "$USERNAME"
print "$USERNAME:$USER_PASSWORD" | chpasswd
sed -i 's/^# %wheel ALL=(ALL:ALL) ALL/%wheel ALL=(ALL:ALL) ALL/' /etc/sudoers

systemctl enable NetworkManager
systemctl enable iwd

mkdir -p /etc/NetworkManager/system-connections
cat >/etc/NetworkManager/system-connections/wifi.nmconnection <<NMC
[connection]
id=wifi
type=wifi
autoconnect=true

[wifi]
ssid=$WIFI_SSID
mode=infrastructure

[wifi-security]
key-mgmt=wpa-psk
psk=$WIFI_PSK

[ipv4]
method=auto

[ipv6]
method=auto
NMC
chmod 600 /etc/NetworkManager/system-connections/wifi.nmconnection

bootctl install
ROOT_UUID=\$(blkid -s UUID -o value "$ROOT_PART")

cat >/boot/loader/loader.conf <<L
default arch
timeout 3
editor no
L

cat >/boot/loader/entries/arch.conf <<E
title   Arch Linux
linux   /vmlinuz-linux
initrd  /initramfs-linux.img
options root=UUID=\$ROOT_UUID rw
E
EOF

print "==> Nettoyage..."
umount -R /mnt

print "==> Installation terminée. Reboot possible."
